/***********************************************************************************************
 * @file    BC3603test.c
 * @version $Rev:: 4837         $
 * @date    $Date:: 2022-09-29 #$
 * @brief   This file provides all the RF firmware functions.
 ***********************************************************************************************
 * @attention
 *
 * Firmware Disclaimer Information
 *
 * 1. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, which is supplied by Holtek Semiconductor Inc., (hereinafter referred to as "HOLTEK") is the
 *    proprietary and confidential intellectual property of HOLTEK, and is protected by copyright law and
 *    other intellectual property laws.
 *
 * 2. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, is confidential information belonging to HOLTEK, and must not be disclosed to any third parties
 *    other than HOLTEK and the customer.
 *
 * 3. The program technical documentation, including the code, is provided "as is" and for customer reference
 *    only. After delivery by HOLTEK, the customer shall use the program technical documentation, including
 *    the code, at their own risk. HOLTEK disclaims any expressed, implied or statutory warranties, including
 *    the warranties of merchantability, satisfactory quality and fitness for a particular purpose.
 *
 * <h2><center>Copyright (C) Holtek Semiconductor Inc. All rights reserved</center></h2>
 ***********************************************************************************************/

/* Includes -----------------------------------------------------------------------------------*/
#include	"ht32.h"
#include "bc3603.h"
#include	"bc3603spi.h"
#include	"bc3603test.h"

#define	DSPI_CSN_PORT				HT_GPIOA
#define  DSPI_CSN   					GPIO_PIN_7					/*PA7*/
#define	DSPI_CSN_AFIO_PORT		GPIO_PA
#define	DSPI_CSN_AFIO_FUN			AFIO_FUN_SPI

#define	DSPI_SCK_PORT				HT_GPIOA
#define  DSPI_SCK   					GPIO_PIN_4					/*PA4*/
#define	DSPI_SCK_AFIO_PORT		GPIO_PA
#define	DSPI_SCK_AFIO_FUN			AFIO_FUN_SPI

#define	DSPI_MOSI_PORT				HT_GPIOA
#define  DSPI_MOSI   				GPIO_PIN_5					/*PA5*/
#define	DSPI_MOSI_AFIO_PORT		GPIO_PA
#define	DSPI_MOSI_AFIO_FUN		AFIO_FUN_SPI

#define	DSPI_MISO_PORT				HT_GPIOA
#define  DSPI_MISO   				GPIO_PIN_6					/*PA6*/
#define	DSPI_MISO_AFIO_PORT		GPIO_PA
#define	DSPI_MISO_AFIO_FUN		AFIO_FUN_SPI

#define  DSPI_PORT         		HT_SPI0
#define	DSPI_PCLK					CKCU_PCLK_SPI0
#define	DSPI_IRQn					SPI0_IRQn
#define	DSPI_IRQHandler			SPI0_IRQHandler
#define	DSPI_ACTIVE					(DSPI_PORT->CR0 |= SPI_SEL_ACTIVE)
#define	DSPI_INACTIVE 				(DSPI_PORT->CR0 &= SPI_SEL_INACTIVE)

#define	TRXD_GIO						RF_GIO2
#define	TRXD_GIO_FUN            RF_GIO_IRQ
#define	TBCLK_GIO					RF_GIO3
#define	TBCLK_GIO_FUN           RF_GIO_INPUT


static u8 *TRxBufferAddr;
static u32 TRxBufferSize,TargetRead,TargetWrite;

static bool	DirTermination,DirectMode;

uc8	PN9Patten[] = 
{
0xFF,0x83,0xDF,0x17,0x32,0x09,0x4E,0xD1,0xE7,0xCD,0x8A,0x91,0xC6,0xD5,0xC4,0xC4,
0x40,0x21,0x18,0x4E,0x55,0x86,0xF4,0xDC,0x8A,0x15,0xA7,0xEC,0x92,0xDF,0x93,0x53,
0x30,0x18,0xCA,0x34,0xBF,0xA2,0xC7,0x59,0x67,0x8F,0xBA,0x0D,0x6D,0xD8,0x2D,0x7D,
0x54,0x0A,0x57,0x97,0x70,0x39,0xD2,0x7A,0xEA,0x24,0x33,0x85,0xED,0x9A,0x1D,0xE1,
0xFF,0x07,0xBE,0x2E,0x64,0x12,0x9D,0xA3,0xCF,0x9B,0x15,0x23,0x8D,0xAB,0x89,0x88,
0x80,0x42,0x30,0x9C,0xAB,0x0D,0xE9,0xB9,0x14,0x2B,0x4F,0xD9,0x25,0xBF,0x26,0xA6,
0x60,0x31,0x94,0x69,0x7F,0x45,0x8E,0xB2,0xCF,0x1F,0x74,0x1A,0xDB,0xB0,0x5A,0xFA,
0xA8,0x14,0xAF,0x2E,0xE0,0x73,0xA4,0xF5,0xD4,0x48,0x67,0x0B,0xDB,0x34,0x3B,0xC3,
0xFE,0x0F,0x7C,0x5C,0xC8,0x25,0x3B,0x47,0x9F,0x36,0x2A,0x47,0x1B,0x57,0x13,0x11,
0x00,0x84,0x61,0x39,0x56,0x1B,0xD3,0x72,0x28,0x56,0x9F,0xB2,0x4B,0x7E,0x4D,0x4C,
0xC0,0x63,0x28,0xD2,0xFE,0x8B,0x1D,0x65,0x9E,0x3E,0xE8,0x35,0xB7,0x60,0xB5,0xF5,
0x50,0x29,0x5E,0x5D,0xC0,0xE7,0x49,0xEB,0xA8,0x90,0xCE,0x17,0xB6,0x68,0x77,0x87,
0xFC,0x1E,0xF8,0xB9,0x90,0x4A,0x76,0x8F,0x3E,0x6C,0x54,0x8E,0x36,0xAE,0x26,0x22,
0x01,0x08,0xC2,0x72,0xAC,0x37,0xA6,0xE4,0x50,0xAD,0x3F,0x64,0x96,0xFC,0x9A,0x99,
0x80,0xC6,0x51,0xA5,0xFD,0x16,0x3A,0xCB,0x3C,0x7D,0xD0,0x6B,0x6E,0xC1,0x6B,0xEA,
0xA0,0x52,0xBC,0xBB,0x81,0xCE,0x93,0xD7,0x51,0x21,0x9C,0x2F,0x6C,0xD0,0xEF,0x0F,
0xF8,0x3D,0xF1,0x73,0x20,0x94,0xED,0x1E,0x7C,0xD8,0xA9,0x1C,0x6D,0x5C,0x4C,0x44,
0x02,0x11,0x84,0xE5,0x58,0x6F,0x4D,0xC8,0xA1,0x5A,0x7E,0xC9,0x2D,0xF9,0x35,0x33,
0x01,0x8C,0xA3,0x4B,0xFA,0x2C,0x75,0x96,0x78,0xFB,0xA0,0xD6,0xDD,0x82,0xD7,0xD5,
0x40,0xA5,0x79,0x77,0x03,0x9D,0x27,0xAE,0xA2,0x43,0x38,0x5E,0xD9,0xA1,0xDE,0x1F,
0xF0,0x7B,0xE2,0xE6,0x41,0x29,0xDA,0x3C,0xF9,0xB1,0x52,0x38,0xDA,0xB8,0x98,0x88,
0x04,0x23,0x09,0xCA,0xB0,0xDE,0x9B,0x91,0x42,0xB4,0xFD,0x92,0x5B,0xF2,0x6A,0x66,
0x03,0x19,0x46,0x97,0xF4,0x58,0xEB,0x2C,0xF1,0xF7,0x41,0xAD,0xBB,0x05,0xAF,0xAA,
0x81,0x4A,0xF2,0xEE,0x07,0x3A,0x4F,0x5D,0x44,0x86,0x70,0xBD,0xB3,0x43,0xBC,0x3F,
0xE0,0xF7,0xC5,0xCC,0x82,0x53,0xB4,0x79,0xF3,0x62,0xA4,0x71,0xB5,0x71,0x31,0x10,
0x08,0x46,0x13,0x95,0x61,0xBD,0x37,0x22,0x85,0x69,0xFB,0x24,0xB7,0xE4,0xD4,0xCC,
0x06,0x32,0x8D,0x2F,0xE8,0xB1,0xD6,0x59,0xE3,0xEE,0x83,0x5B,0x76,0x0B,0x5F,0x55,
0x02,0x95,0xE5,0xDC,0x0E,0x74,0x9E,0xBA,0x89,0x0C,0xE1,0x7B,0x66,0x87,0x78,0x7F,
0xC1,0xEF,0x8B,0x99,0x04,0xA7,0x68,0xF3,0xE6,0xC5,0x48,0xE3,0x6A,0xE2,0x62,0x20,
0x10,0x8C,0x27,0x2A,0xC3,0x7A,0x6E,0x45,0x0A,0xD3,0xF6,0x49,0x6F,0xC9,0xA9,0x98,
0x0C,0x65,0x1A,0x5F,0xD1,0x63,0xAC,0xB3,0xC7,0xDD,0x06,0xB6,0xEC,0x16,0xBE,0xAA,
0x05,0x2B,0xCB,0xB8,0x1C,0xE9,0x3D,0x75,0x12,0x19,0xC2,0xF6,0xCD,0x0E,0xF0   
};

/***********************************************************************************************
 * @brief Configure the direct mode of RF.
 * @param DirMode: Selection of direct mode
 *   This parameter can be one of below:
 *     @arg CARRIER 	: Tx mode of carrier 
 *     @arg DIRECT_TX: Tx mode 
 *     @arg DIRECT_RX: Rx mode
 * @retval None
 ***********************************************************************************************/
static void BC3603_TestDelay(u32 Delay)
{
	while(Delay--) __NOP();
}
/***********************************************************************************************
 * @brief Configure the direct mode of RF.
 * @param DirMode: Selection of direct mode
 *   This parameter can be one of below:
 *     @arg CARRIER 	: Tx mode of carrier 
 *     @arg DIRECT_TX: Tx mode 
 *     @arg DIRECT_RX: Rx mode
 * @retval None
 ***********************************************************************************************/
void BC3603_DirectModeActive(RF_DIRM_Enum DirMode) 
{
	u8		reg;
	u8		trx;
	
	switch(DirMode)
	{
		case CARRIER:		/* carrier mode */
			BC3603_RegisterBank(REGS_BANK1);
			BC3603_WriteRegister(0x2D,0x00);					/* Frequency deviation = 0 */
			BC3603_WriteRegister(0x2E,0x00);           	/* Frequency deviation = 0 */
			BC3603_RegisterBank(REGS_BANK0);
			trx = OM_RTX_SEL;									/* Tx mode */
			break;
		case DIRECT_TX:	/* direct Tx mode */
			BC3603_SetGIOFunction(TRXD_GIO,RF_GIO_TRXD);		/* GIO to TXD input*/
			BC3603_SetGIOFunction(TBCLK_GIO,RF_GIO_TBCLK);	/* GIO to TBCLK output */
			trx = OM_RTX_SEL;									/* Tx mode */
			break;
		case DIRECT_RX:	/* direct Rx mode */
			BC3603_SetGIOFunction(TRXD_GIO,RF_GIO_TRXD);		/* GIO to RXD output*/
			BC3603_SetGIOFunction(TBCLK_GIO,RF_GIO_RBCLK);	/* GIO to RBCLK output */
			trx = 0x00;												/* Rx mode */
			break;
	}
	/* VCO calibration */
	BC3603_VCOAutoCalibration();
	BC3603_ClearIRQFlag(BC3603_GetIRQFlag());
	/* Enable direct mode */
	BC3603_WriteRegister(CONFIG_REGS,BC3603_ReadRegister(CONFIG_REGS) | CFG_DIR_EN);

	/* Tx/Rx select & Synthesizer enable & Tx/Rx enable */	
	reg = BC3603_ReadRegister(OPERATION_CTL_REGS);
	reg |= (OM_SX_EN | trx);								
	BC3603_WriteRegister(OPERATION_CTL_REGS,reg);
	BC3603_TestDelay(30);
	reg |= OM_RTX_EN;
	BC3603_WriteRegister(OPERATION_CTL_REGS,reg);
}
/***********************************************************************************************
 * @brief Direct mode to stop RF.
 * @param None
 * @retval None
 ***********************************************************************************************/
void BC3603_DirectModeInactive(void)
{
	u8	reg;
	/* Synthesizer disable & Tx/Rx disable */	
	reg = BC3603_ReadRegister(OPERATION_CTL_REGS);
	reg &= ~OM_RTX_EN;
	BC3603_WriteRegister(OPERATION_CTL_REGS,reg);	
	BC3603_TestDelay(30);
	reg &= ~OM_SX_EN;
	BC3603_WriteRegister(OPERATION_CTL_REGS,reg);	
	/* Disable direct mode */
	BC3603_WriteRegister(CONFIG_REGS,BC3603_ReadRegister(CONFIG_REGS) & ~CFG_DIR_EN);
	BC3603_SetGIOFunction(TRXD_GIO,TRXD_GIO_FUN);		/* Restore GIO functionality */
	BC3603_SetGIOFunction(TBCLK_GIO,TBCLK_GIO_FUN);		/* Restore GIO functionality */
}
/***********************************************************************************************
 * @brief Config Direct Transmit Data of SPI
 * @param None
 * @retval None
 ***********************************************************************************************/
void BC3603_DirectTransmit(u32 DataRate,u8 *SAddr,u16 SPos,u16 Size)
{
	SPI_InitTypeDef	SPI_InitStructure;
	
   TRxBufferAddr = SAddr;
   TRxBufferSize = Size;
	TargetRead = SPos;
	DirectMode = TRUE;						/* Tx mode */
	DirTermination = FALSE;

	/* Deinitialize the SPI peripheral registers */
	SPI_DeInit(DSPI_PORT);	
	/* SPI Configuration */	
	SPI_InitStructure.SPI_Mode = SPI_SLAVE;
	SPI_InitStructure.SPI_FIFO = SPI_FIFO_ENABLE;
	SPI_InitStructure.SPI_DataLength = SPI_DATALENGTH_8;
	SPI_InitStructure.SPI_SELMode = SPI_SEL_SOFTWARE;
	SPI_InitStructure.SPI_SELPolarity = SPI_SELPOLARITY_HIGH;
	SPI_InitStructure.SPI_FirstBit = SPI_FIRSTBIT_MSB;
	SPI_InitStructure.SPI_CPOL = SPI_CPOL_LOW;
	SPI_InitStructure.SPI_CPHA = SPI_CPHA_SECOND;
	SPI_InitStructure.SPI_RxFIFOTriggerLevel = 0;
	SPI_InitStructure.SPI_TxFIFOTriggerLevel = SPI_DATALENGTH_4;
	SPI_InitStructure.SPI_ClockPrescaler = CKCU_GetPeripFrequency(DSPI_PCLK)/DataRate;
	SPI_Init(DSPI_PORT, &SPI_InitStructure);	
   /* Disable the SEL output for the specified SPI peripheral.*/
	SPI_SELOutputCmd(DSPI_PORT,DISABLE);				
   /* Enable the selected SPI peripheral */
	SPI_Cmd(DSPI_PORT,ENABLE);	
	/* configure CSN/SCK/MISO as SPI  */   		
   AFIO_GPxConfig(DSPI_CSN_AFIO_PORT,DSPI_CSN,AFIO_FUN_SPI);
   AFIO_GPxConfig(DSPI_SCK_AFIO_PORT,DSPI_SCK,AFIO_FUN_SPI);
   AFIO_GPxConfig(DSPI_MISO_AFIO_PORT,DSPI_MISO,AFIO_FUN_SPI);	
	/*-- Enable or Disable the specified SPI interrupt --*/
	SPI_IntConfig(DSPI_PORT,SPI_INT_TXBE,ENABLE);
	/* Enable interrupt */
   NVIC_EnableIRQ(DSPI_IRQn);
}
/***********************************************************************************************
 * @brief Config Direct Transmit Data of SPI
 * @param None
 * @retval None
 ***********************************************************************************************/
void BC3603_DirectTxTermination(void)
{
	u8	x;
	
	DirTermination = TRUE;
	x=200;
	while(DirTermination && (x != 0)) {	BC3603_TestDelay(10000); 	x--; }

	/* Disable interrupt */	
	NVIC_DisableIRQ(DSPI_IRQn);
	/*-- Enable or Disable the specified SPI interrupt --*/	
	SPI_IntConfig(DSPI_PORT,SPI_INT_ALL,DISABLE);				
   /* Disable the selected SPI peripheral */
	SPI_Cmd(DSPI_PORT,DISABLE);
	
   AFIO_GPxConfig(DSPI_CSN_AFIO_PORT,DSPI_CSN,AFIO_FUN_GPIO);
   AFIO_GPxConfig(DSPI_SCK_AFIO_PORT,DSPI_SCK,AFIO_FUN_GPIO);
   AFIO_GPxConfig(DSPI_MISO_AFIO_PORT,DSPI_MISO,AFIO_FUN_GPIO);			
}
/***********************************************************************************************
 * @brief Config Direct Receiver Data of SPI
 * @param None
 * @retval None
 ***********************************************************************************************/
void BC3603_DirectReceiver(u32 DataRate,u8 *SAddr,u16 Size)
{
	SPI_InitTypeDef	SPI_InitStructure;
	
   TRxBufferAddr = SAddr;
   TRxBufferSize = Size;
	TargetRead = 0;
	TargetWrite = 0;	
	DirectMode = FALSE;							/* Rx mode */
	DirTermination = FALSE;

	/* Deinitialize the SPI peripheral registers */
	SPI_DeInit(DSPI_PORT);	
	/* SPI Configuration */	
	SPI_InitStructure.SPI_Mode = SPI_SLAVE;
	SPI_InitStructure.SPI_FIFO = SPI_FIFO_ENABLE;
	SPI_InitStructure.SPI_DataLength = SPI_DATALENGTH_8;
	SPI_InitStructure.SPI_SELMode = SPI_SEL_SOFTWARE;
	SPI_InitStructure.SPI_SELPolarity = SPI_SELPOLARITY_HIGH;
	SPI_InitStructure.SPI_FirstBit = SPI_FIRSTBIT_MSB;
	SPI_InitStructure.SPI_CPOL = SPI_CPOL_LOW;
	SPI_InitStructure.SPI_CPHA = SPI_CPHA_FIRST;
	SPI_InitStructure.SPI_RxFIFOTriggerLevel = SPI_DATALENGTH_4;
	SPI_InitStructure.SPI_TxFIFOTriggerLevel = 0;
	SPI_InitStructure.SPI_ClockPrescaler = CKCU_GetPeripFrequency(DSPI_PCLK)/DataRate;
	SPI_Init(DSPI_PORT, &SPI_InitStructure);	
   /* Disable the SEL output for the specified SPI peripheral.*/
	SPI_SELOutputCmd(DSPI_PORT,DISABLE);				
   /* Enable the selected SPI peripheral */
	SPI_Cmd(DSPI_PORT,ENABLE);	
   /* clear SPI FIFO buffer */	
	while(SPI_GetFIFOStatus(DSPI_PORT,SPI_FIFO_RX)) Size = DSPI_PORT->DR;
	/* configure CSN/SCK/MOSI as SPI  */   	
   AFIO_GPxConfig(DSPI_CSN_AFIO_PORT,DSPI_CSN,AFIO_FUN_SPI);
   AFIO_GPxConfig(DSPI_SCK_AFIO_PORT,DSPI_SCK,AFIO_FUN_SPI);
   AFIO_GPxConfig(DSPI_MOSI_AFIO_PORT,DSPI_MOSI,AFIO_FUN_SPI);	
	/*-- Enable or Disable the specified SPI interrupt --*/
	SPI_IntConfig(DSPI_PORT,SPI_INT_RXBNE,ENABLE);
	/* Enable interrupt */
   NVIC_EnableIRQ(DSPI_IRQn);
}
/***********************************************************************************************
 * @brief Get Receiver Data
 * @param None
 * @retval None
 ***********************************************************************************************/
bool BC3603_GetDirectReceiverData(u8 *Data)
{
	if(TargetRead != TargetWrite)
	{
		*Data = *(TRxBufferAddr + TargetRead);
		if(++TargetRead >= TRxBufferSize)	TargetRead = 0;
		return(TRUE);
	}
	return(FALSE);
}
/***********************************************************************************************
 * @brief Config Direct Transmit Data of SPI
 * @param None
 * @retval None
 ***********************************************************************************************/
void BC3603_DirectRxTermination(void)
{
	
	DirTermination = TRUE;
	while(DirTermination && SPI_GetFIFOStatus(DSPI_PORT,SPI_FIFO_RX));
	if(DirTermination)
	{
		/* Disable interrupt */	
		NVIC_DisableIRQ(DSPI_IRQn);
		/*-- Enable or Disable the specified SPI interrupt --*/	
		SPI_IntConfig(DSPI_PORT,SPI_INT_ALL,DISABLE);
		DirTermination = FALSE;
	}
   /* Disable the selected SPI peripheral */
	SPI_Cmd(DSPI_PORT,DISABLE);
	
   AFIO_GPxConfig(DSPI_CSN_AFIO_PORT,DSPI_CSN,AFIO_FUN_GPIO);
   AFIO_GPxConfig(DSPI_SCK_AFIO_PORT,DSPI_SCK,AFIO_FUN_GPIO);
   AFIO_GPxConfig(DSPI_MOSI_AFIO_PORT,DSPI_MOSI,AFIO_FUN_GPIO);
}
/***********************************************************************************************
 * @brief Config Direct Transmit Data of SPI
 * @param None
 * @retval None
 ***********************************************************************************************/
void DSPI_IRQHandler(void)
{
	u8	x;
	
	if(DirectMode)
	{   
		if(DSPI_PORT->SR & SPI_FLAG_TXBE)
		{
			if(!DirTermination)
			{
				x = 8 - SPI_GetFIFOStatus(DSPI_PORT,SPI_FIFO_TX);			
				while(x)
				{
					DSPI_PORT->DR = *(TRxBufferAddr+TargetRead);
					if(++TargetRead >= TRxBufferSize)	TargetRead = 0;
					x--;
				}
			}
			else
			{
				SPI_IntConfig(DSPI_PORT,SPI_INT_ALL,DISABLE);
				while(SPI_GetFIFOStatus(DSPI_PORT,SPI_FIFO_TX));
				while(!(DSPI_PORT->SR & SPI_FLAG_TXE));
				NVIC_DisableIRQ(DSPI_IRQn);
				DirTermination = FALSE;
			}
		}
	}
	else
	{
		if(DSPI_PORT->SR & SPI_FLAG_RXBNE)
		{
			if(DirTermination)
			{
				SPI_IntConfig(DSPI_PORT,SPI_INT_ALL,DISABLE);				
				NVIC_DisableIRQ(DSPI_IRQn);				
				DirTermination = FALSE;
			}
			x = SPI_GetFIFOStatus(DSPI_PORT,SPI_FIFO_RX);
			while(x)
			{
				*(TRxBufferAddr+TargetWrite) = DSPI_PORT->DR;
				if(++TargetWrite >= TRxBufferSize)	TargetWrite = 0;
				x--;
			}
		}		
	}	
}
